# Amazon Q Business Deployment Guide

This guide provides two approaches to deploy Amazon Q Business: Terraform and AWS CLI.

## Prerequisites Check

Before proceeding, ensure you have:

- [x] AWS CLI v2 installed and configured
- [x] Valid AWS credentials with administrator access
- [x] Terraform v1.9.8 installed
- [x] jq (JSON processor) installed
- [ ] **AWS IAM Identity Center enabled** (REQUIRED!)

## Current Status

Your environment:
- AWS CLI: ✓ Installed (v2.31.22)
- AWS Credentials: ✓ Configured (Account: 460783431756, Region: eu-central-1)
- Terraform: ✓ Installed (v1.9.8)
- jq: ✓ Installed (v1.7)
- **IAM Identity Center: ✗ NOT ENABLED**

## CRITICAL: Enable IAM Identity Center First

Amazon Q Business **requires** AWS IAM Identity Center. You must enable it before deployment:

### Steps to Enable IAM Identity Center:

1. **Open AWS Console**
   - Go to: https://console.aws.amazon.com/singlesignon
   - Or search for "IAM Identity Center" in the AWS Console

2. **Enable Identity Center**
   - Click "Enable" button
   - Choose organization management (new or existing)
   - Select your preferred identity source (Identity Center directory, Active Directory, or External IdP)

3. **Complete Setup**
   - Follow the setup wizard
   - Create at least one user (you'll need this for Q Business access)
   - Note the Identity Center instance ARN (you'll need it for deployment)

4. **Verify Setup**
   - Run: `./get-identity-center-arn.sh`
   - You should see your Identity Center instance ARN

## Deployment Options

### Option 1: Quick Deployment with Bash Script (Recommended)

The fastest way to deploy if you have Identity Center enabled:

```bash
# Run the deployment script
./deploy.sh

# Or with custom app name and region
./deploy.sh my-app-name us-east-1
```

This script will:
- Check prerequisites
- Verify IAM Identity Center is enabled
- Create necessary IAM roles
- Create Amazon Q Business application
- Create an index
- Create a web experience
- Save all configuration to /tmp/*.json files

### Option 2: Terraform Deployment

For infrastructure-as-code approach:

1. **Get your Identity Center ARN**
   ```bash
   ./get-identity-center-arn.sh
   ```

2. **Update terraform.tfvars**
   ```hcl
   identity_center_instance_arn = "arn:aws:sso:::instance/ssoins-xxxxx"
   application_name             = "my-qbusiness-app"
   aws_region                  = "eu-central-1"
   ```

3. **Deploy with Terraform**
   ```bash
   # Already initialized
   terraform plan
   terraform apply
   ```

## What Gets Created

Both deployment methods create:

1. **IAM Role**: `{app-name}-app-role`
   - Allows Amazon Q Business to assume the role
   - Permissions for Q Business and S3 access

2. **Amazon Q Business Application**
   - Enterprise search and chat application
   - Integrated with IAM Identity Center

3. **Index**
   - ENTERPRISE type
   - 1 capacity unit (configurable)

4. **Web Experience**
   - Chat interface for users
   - Accessible via web URL

## After Deployment

### 1. Get Your Web Experience URL

```bash
jq -r '.defaultEndpoint' /tmp/qbusiness-web-experience.json
```

### 2. Configure User Access

Users must be granted access via IAM Identity Center:

```bash
APP_ID=$(jq -r '.applicationId' /tmp/qbusiness-app.json)

# List available users from Identity Center
aws identitystore list-users \
  --identity-store-id <your-identity-store-id>

# Grant user access to Q Business
aws qbusiness put-user-access-grant \
  --application-id $APP_ID \
  --user-id <user-id> \
  --region eu-central-1
```

### 3. Add Data Sources

Configure data sources to index your content:

```bash
# Example: Create S3 data source
aws qbusiness create-data-source \
  --application-id $APP_ID \
  --index-id <index-id> \
  --display-name "My S3 Documents" \
  --type S3 \
  --configuration file://s3-datasource-config.json \
  --region eu-central-1
```

### 4. Start Using Q Business

1. Navigate to the web experience URL
2. Sign in with your IAM Identity Center credentials
3. Start asking questions

## Cleanup

### If using Terraform:
```bash
terraform destroy
```

### If using CLI script:
```bash
# Delete web experience
aws qbusiness delete-web-experience \
  --application-id <app-id> \
  --web-experience-id <web-exp-id>

# Delete index
aws qbusiness delete-index \
  --application-id <app-id> \
  --index-id <index-id>

# Delete application
aws qbusiness delete-application \
  --application-id <app-id>

# Delete IAM role
aws iam delete-role-policy \
  --role-name <role-name> \
  --policy-name <policy-name>
aws iam delete-role --role-name <role-name>
```

## Troubleshooting

### "Identity Center not enabled" Error
- Enable IAM Identity Center in AWS Console first
- Run `./get-identity-center-arn.sh` to verify

### "Region not supported" Error
Amazon Q Business is available in limited regions:
- us-east-1 (US East - N. Virginia)
- us-west-2 (US West - Oregon)
- Check AWS documentation for current region availability

### "Access Denied" Errors
- Ensure your AWS credentials have administrator access
- Required permissions:
  - `qbusiness:*`
  - `iam:CreateRole`, `iam:PutRolePolicy`
  - `sso:*`, `sso-admin:*`
  - `identitystore:*`

### "Application creation failed"
- Check CloudWatch Logs for detailed error messages
- Verify IAM role trust policy allows qbusiness.amazonaws.com

## Pricing

Amazon Q Business pricing (as of 2024):
- **Lite tier**: $3 per user per month
- **Business tier**: $20 per user per month
- Additional charges for:
  - Index storage
  - Document ingestion
  - API calls

Always check current pricing: https://aws.amazon.com/q/business/pricing/

## Support

For issues:
1. Check CloudWatch Logs for Q Business
2. Review AWS documentation: https://docs.aws.amazon.com/amazonq/
3. AWS Support: https://console.aws.amazon.com/support

## Next Steps

1. **Enable IAM Identity Center** (if not already done)
2. **Run deployment** using your preferred method
3. **Configure users** in Identity Center
4. **Add data sources** (S3, SharePoint, databases, etc.)
5. **Test the web experience**
6. **Monitor usage** and costs

## Files in This Directory

- `main.tf` - Terraform main configuration
- `variables.tf` - Terraform variables
- `outputs.tf` - Terraform outputs
- `terraform.tfvars` - Deployment configuration
- `deploy.sh` - Quick deployment script
- `get-identity-center-arn.sh` - Helper to get Identity Center ARN
- `README.md` - Terraform-specific readme
- `DEPLOYMENT_GUIDE.md` - This file

---

**Remember**: IAM Identity Center must be enabled before any deployment will work!
