# ============================================
# Terraform Archive Provider 使用示例
# ============================================

# 在 main.tf 中添加 archive provider
# terraform {
#   required_providers {
#     archive = {
#       source  = "hashicorp/archive"
#       version = "~> 2.4"
#     }
#   }
# }

# ============================================
# 示例 1: 归档单个文件
# ============================================
data "archive_file" "single_file" {
  type        = "zip"
  source_file = "${path.module}/qbusiness-interactive.sh"
  output_path = "${path.module}/archives/interactive-script.zip"
}

# ============================================
# 示例 2: 归档整个目录
# ============================================
data "archive_file" "terraform_config" {
  type        = "zip"
  source_dir  = "${path.module}"
  output_path = "${path.module}/archives/terraform-config.zip"

  # 排除特定文件/目录
  excludes = [
    ".terraform",
    "*.tfstate",
    "*.tfstate.backup",
    ".git",
    "archives"
  ]
}

# ============================================
# 示例 3: 归档多个指定文件
# ============================================
data "archive_file" "documentation" {
  type        = "zip"
  output_path = "${path.module}/archives/documentation.zip"

  source {
    content  = file("${path.module}/README.md")
    filename = "README.md"
  }

  source {
    content  = file("${path.module}/DEPLOYMENT_GUIDE.md")
    filename = "DEPLOYMENT_GUIDE.md"
  }

  source {
    content  = file("${path.module}/SETUP_COMPLETE.md")
    filename = "SETUP_COMPLETE.md"
  }
}

# ============================================
# 示例 4: 使用 tar.gz 格式
# ============================================
# Commented out - tar type not supported by archive provider
# data "archive_file" "backup_targz" {
#   type        = "tar"
#   source_dir  = "${path.module}"
#   output_path = "${path.module}/archives/backup.tar.gz"
#
#   excludes = [
#     ".terraform",
#     "*.tfstate",
#     "archives"
#   ]
# }

# ============================================
# 示例 5: Lambda 函数打包示例
# ============================================
# 假设你有一个 Lambda 函数代码
# Commented out - lambda directory does not exist
# data "archive_file" "lambda_function" {
#   type        = "zip"
#   source_dir  = "${path.module}/lambda"  # Lambda 代码目录
#   output_path = "${path.module}/archives/lambda-function.zip"
# }

# 可以配合 S3 使用
# resource "aws_s3_object" "lambda_zip" {
#   bucket = "my-lambda-bucket"
#   key    = "lambda-function-${data.archive_file.lambda_function.output_md5}.zip"
#   source = data.archive_file.lambda_function.output_path
#   etag   = data.archive_file.lambda_function.output_md5
# }

# ============================================
# 示例 6: 动态内容归档
# ============================================
data "archive_file" "deployment_manifest" {
  type        = "zip"
  output_path = "${path.module}/archives/deployment-manifest.zip"

  source {
    content  = jsonencode({
      deployment_date = timestamp()
      application_name = var.application_name
      region = var.aws_region
      version = "1.0.0"
    })
    filename = "manifest.json"
  }

  source {
    content  = templatefile("${path.module}/README.md", {})
    filename = "readme.txt"
  }
}

# ============================================
# Outputs - 查看归档信息
# ============================================
output "archive_outputs" {
  description = "Archive file information"
  value = {
    terraform_config_hash = data.archive_file.terraform_config.output_md5
    terraform_config_size = data.archive_file.terraform_config.output_size
    terraform_config_path = data.archive_file.terraform_config.output_path
  }
}
