#!/bin/bash
# Amazon Q Business Interactive Chat Session with SSO
# 使用 SSO 的交互式聊天会话

set -e

APP_ID="8ea6e006-85cc-47bd-a587-b14bff19884d"
REGION="us-west-2"
PROFILE="qbusiness-sso"
CONVERSATION_ID=""
PARENT_MESSAGE_ID=""

# 颜色定义
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

echo -e "${GREEN}======================================${NC}"
echo -e "${GREEN}Amazon Q Business 交互式会话 (SSO)${NC}"
echo -e "${GREEN}======================================${NC}"

# 检查 SSO 登录状态
echo -e "${BLUE}检查 SSO 登录状态...${NC}"
if ! aws sts get-caller-identity --profile $PROFILE &>/dev/null; then
    echo -e "${RED}错误: SSO 未登录${NC}"
    echo ""
    echo -e "${YELLOW}请先运行以下命令登录:${NC}"
    echo "  aws sso login --profile $PROFILE"
    echo ""
    echo -e "${YELLOW}登录 URL:${NC}"
    echo "  https://d-c367615483.awsapps.com/start"
    echo ""
    exit 1
fi

echo -e "${GREEN}✓ SSO 已登录${NC}"
echo ""

echo -e "${YELLOW}命令:${NC}"
echo "  输入问题进行对话"
echo "  'exit' 或 'quit' - 退出"
echo "  'clear' - 清空会话历史"
echo "  'help' - 显示帮助"
echo "  'status' - 显示当前会话状态"
echo -e "${GREEN}======================================${NC}"
echo ""

# 发送消息函数
send_message() {
    local message="$1"
    local params="--application-id $APP_ID --region $REGION --profile $PROFILE"

    # 如果有会话 ID，添加到参数中
    if [ -n "$CONVERSATION_ID" ]; then
        params="$params --conversation-id $CONVERSATION_ID"
    fi

    if [ -n "$PARENT_MESSAGE_ID" ]; then
        params="$params --parent-message-id $PARENT_MESSAGE_ID"
    fi

    # 发送消息并解析响应
    local response=$(aws qbusiness chat-sync \
        $params \
        --user-message "$message" \
        2>&1)

    if [ $? -eq 0 ]; then
        # 保存会话 ID 以保持上下文
        CONVERSATION_ID=$(echo "$response" | jq -r '.conversationId // empty')
        PARENT_MESSAGE_ID=$(echo "$response" | jq -r '.systemMessageId // empty')

        # 显示回复
        echo -e "${BLUE}Q Business:${NC}"
        echo "$response" | jq -r '.systemMessage // .failedAttachments // "No response"'

        # 显示引用的文档（如果有）
        local sources=$(echo "$response" | jq -r '.sourceAttributions[]?.title // empty' 2>/dev/null)
        if [ -n "$sources" ]; then
            echo -e "\n${YELLOW}参考文档:${NC}"
            echo "$response" | jq -r '.sourceAttributions[] | "  - \(.title) [\(.snippet // "")]"'
        fi
    else
        echo -e "${RED}错误: 无法发送消息${NC}"
        echo "$response"

        # 检查是否是 SSO token 过期
        if echo "$response" | grep -q "SSO Token"; then
            echo ""
            echo -e "${YELLOW}SSO token 可能已过期，请重新登录:${NC}"
            echo "  aws sso login --profile $PROFILE"
        fi
    fi

    echo ""
}

# 显示帮助
show_help() {
    echo -e "${YELLOW}可用命令:${NC}"
    echo "  exit/quit  - 退出聊天"
    echo "  clear      - 清空会话历史（开始新对话）"
    echo "  help       - 显示此帮助信息"
    echo "  status     - 显示当前会话状态"
    echo ""
}

# 显示状态
show_status() {
    echo -e "${YELLOW}当前会话状态:${NC}"
    echo "  Application ID: $APP_ID"
    echo "  Region: $REGION"
    echo "  SSO Profile: $PROFILE"
    echo "  Conversation ID: ${CONVERSATION_ID:-'(new session)'}"

    # 显示当前 SSO 用户
    identity=$(aws sts get-caller-identity --profile $PROFILE 2>/dev/null)
    if [ $? -eq 0 ]; then
        echo "  SSO User ARN: $(echo "$identity" | jq -r '.Arn')"
    fi
    echo ""
}

# 主循环
while true; do
    echo -n -e "${GREEN}You:${NC} "
    read -r input

    # 处理空输入
    if [ -z "$input" ]; then
        continue
    fi

    # 处理命令
    case "$input" in
        exit|quit)
            echo -e "${GREEN}再见! 会话已结束。${NC}"
            break
            ;;
        clear)
            CONVERSATION_ID=""
            PARENT_MESSAGE_ID=""
            echo -e "${YELLOW}会话历史已清空，开始新对话${NC}"
            echo ""
            ;;
        help)
            show_help
            ;;
        status)
            show_status
            ;;
        *)
            send_message "$input"
            ;;
    esac
done
