#!/bin/bash
# Amazon Q Business Chat with SSO Authentication
# 使用 SSO 身份验证与 Amazon Q Business 对话

set -e

# 动态获取 Application ID，如果文件不存在则使用默认值
APP_ID=$(jq -r '.applicationId' /tmp/qbusiness-app.json 2>/dev/null || echo "8ea6e006-85cc-47bd-a587-b14bff19884d")
REGION="us-west-2"
PROFILE="qbusiness-sso"

# 颜色定义
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

echo -e "${GREEN}======================================${NC}"
echo -e "${GREEN}Amazon Q Business - SSO 聊天${NC}"
echo -e "${GREEN}======================================${NC}"

# 检查是否提供了问题
if [ -z "$1" ]; then
    echo -e "${YELLOW}用法:${NC} $0 '你的问题'"
    echo ""
    echo -e "${YELLOW}示例:${NC}"
    echo "  $0 'What documents are available?'"
    echo "  $0 'Tell me about CloudSync Pro'"
    exit 1
fi

QUESTION="$1"

echo -e "${BLUE}步骤 1/3:${NC} 检查 SSO 登录状态..."

# 检查 SSO 登录状态
if ! aws sts get-caller-identity --profile $PROFILE &>/dev/null; then
    echo -e "${YELLOW}SSO 未登录。请先登录:${NC}"
    echo ""
    echo -e "${GREEN}运行以下命令登录:${NC}"
    echo "  aws sso login --profile $PROFILE"
    echo ""
    echo -e "${YELLOW}或者在浏览器中访问:${NC}"
    echo "  https://d-c367615483.awsapps.com/start"
    echo ""
    echo -e "${RED}登录后，请重新运行此脚本${NC}"
    exit 1
fi

echo -e "${GREEN}✓ SSO 已登录${NC}"
echo ""

echo -e "${BLUE}步骤 2/3:${NC} 发送问题到 Amazon Q Business..."
echo -e "${YELLOW}问题:${NC} $QUESTION"
echo ""

# 发送聊天请求（不传递 user-id）
response=$(aws qbusiness chat-sync \
    --application-id $APP_ID \
    --user-message "$QUESTION" \
    --region $REGION \
    --profile $PROFILE \
    2>&1)

if [ $? -eq 0 ]; then
    echo -e "${BLUE}步骤 3/3:${NC} 接收并显示回复"
    echo -e "${GREEN}======================================${NC}"
    echo -e "${GREEN}Q Business 回复:${NC}"
    echo -e "${GREEN}======================================${NC}"

    # 显示回复内容
    echo "$response" | jq -r '.systemMessage // "无回复内容"'

    # 保存响应
    echo "$response" > /tmp/qbusiness-chat-response-sso.json

    # 保存会话 ID
    conversation_id=$(echo "$response" | jq -r '.conversationId // empty')
    if [ -n "$conversation_id" ]; then
        echo "$conversation_id" > /tmp/qbusiness-conversation-id-sso.txt
    fi

    # 显示引用的文档
    sources=$(echo "$response" | jq -r '.sourceAttributions[]?.title // empty' 2>/dev/null)
    if [ -n "$sources" ]; then
        echo ""
        echo -e "${YELLOW}参考文档:${NC}"
        echo "$response" | jq -r '.sourceAttributions[] | "  - \(.title)"'
    fi

    echo -e "${GREEN}======================================${NC}"
else
    echo -e "${RED}错误: 无法连接到 Q Business${NC}"
    echo "$response"
    exit 1
fi
