#!/bin/bash

# Amazon Q Business - 快速状态检查
# 检查当前部署的所有关键信息

echo "=========================================="
echo "Amazon Q Business 部署状态检查"
echo "=========================================="
echo ""

# Identity Center 信息
echo "📋 Identity Center 配置"
echo "----------------------------------------"
echo "实例名称: iam_claude"
echo "区域: eu-north-1 (Stockholm)"
echo "Identity Store ID: d-c367615483"
echo ""

# 获取 Identity Center 用户
echo "👤 Identity Center 用户列表"
echo "----------------------------------------"
aws identitystore list-users \
  --identity-store-id d-c367615483 \
  --region eu-north-1 \
  --query 'Users[*].[UserName,UserId,DisplayName,Emails[0].Value]' \
  --output table
echo ""

# Q Business 应用信息
APP_ID=$(cat /tmp/qbusiness-app.json 2>/dev/null | jq -r '.applicationId // "未找到"')
WEB_EXP_ID=$(cat /tmp/qbusiness-web-experience.json 2>/dev/null | jq -r '.webExperienceId // "未找到"')
INDEX_ID=$(cat /tmp/qbusiness-index.json 2>/dev/null | jq -r '.indexId // "未找到"')

echo "🤖 Q Business 应用信息"
echo "----------------------------------------"
echo "应用 ID: $APP_ID"
echo "区域: us-west-2"
echo "Index ID: $INDEX_ID"
echo "Web Experience ID: $WEB_EXP_ID"
echo ""

# 获取应用状态
if [ "$APP_ID" != "未找到" ]; then
    echo "📊 应用详细状态"
    echo "----------------------------------------"
    APP_DETAILS=$(aws qbusiness get-application \
      --application-id "$APP_ID" \
      --region us-west-2 2>/dev/null)

    if [ $? -eq 0 ]; then
        echo "名称: $(echo "$APP_DETAILS" | jq -r '.displayName')"
        echo "状态: $(echo "$APP_DETAILS" | jq -r '.status')"
        echo "创建时间: $(echo "$APP_DETAILS" | jq -r '.createdAt')"
        echo "更新时间: $(echo "$APP_DETAILS" | jq -r '.updatedAt')"
    fi
    echo ""
fi

# 获取 Web Experience 状态和 URL
if [ "$WEB_EXP_ID" != "未找到" ] && [ "$APP_ID" != "未找到" ]; then
    echo "🌐 Web Experience 状态"
    echo "----------------------------------------"
    WEB_INFO=$(aws qbusiness get-web-experience \
      --application-id "$APP_ID" \
      --web-experience-id "$WEB_EXP_ID" \
      --region us-west-2 2>/dev/null)

    if [ $? -eq 0 ]; then
        STATUS=$(echo "$WEB_INFO" | jq -r '.status')
        URL=$(echo "$WEB_INFO" | jq -r '.defaultEndpoint')
        echo "状态: $STATUS"
        echo "访问 URL: $URL"
    else
        echo "无法获取 Web Experience 信息"
    fi
    echo ""
fi

# 获取 Index 统计
if [ "$INDEX_ID" != "未找到" ] && [ "$APP_ID" != "未找到" ]; then
    echo "📚 索引统计信息"
    echo "----------------------------------------"
    aws qbusiness get-index \
      --application-id "$APP_ID" \
      --index-id "$INDEX_ID" \
      --region us-west-2 \
      --query 'indexStatistics' \
      --output json 2>/dev/null | jq '.'
    echo ""
fi

# 检查数据源
if [ "$INDEX_ID" != "未找到" ] && [ "$APP_ID" != "未找到" ]; then
    echo "💾 数据源列表"
    echo "----------------------------------------"
    DATA_SOURCES=$(aws qbusiness list-data-sources \
      --application-id "$APP_ID" \
      --index-id "$INDEX_ID" \
      --region us-west-2 2>/dev/null)

    if [ $? -eq 0 ]; then
        echo "$DATA_SOURCES" | jq -r '.dataSources[] | "- \(.displayName) (ID: \(.dataSourceId)) - 状态: \(.status)"' 2>/dev/null || echo "暂无数据源"
    else
        echo "无法获取数据源信息"
    fi
    echo ""
fi

# 访问说明
echo "=========================================="
echo "🚀 如何访问"
echo "=========================================="
echo ""
echo "1. Web 界面访问:"
if [ "$WEB_EXP_ID" != "未找到" ]; then
    URL=$(cat /tmp/qbusiness-web-experience.json 2>/dev/null | jq -r '.defaultEndpoint // "未找到"')
    echo "   $URL"
else
    echo "   未部署 Web Experience"
fi
echo ""
echo "2. CLI 聊天测试:"
echo "   ./chat-simple.sh 'Hello, what can you help me with?'"
echo ""
echo "3. 交互式会话:"
echo "   ./qbusiness-interactive.sh"
echo ""
echo "=========================================="
echo "⚠️  重要提示"
echo "=========================================="
echo ""
echo "如果您看到 403 Forbidden 错误，需要在 AWS 控制台中:"
echo ""
echo "1. 访问 Amazon Q Business 控制台 (us-west-2):"
echo "   https://us-west-2.console.aws.amazon.com/amazonq/business"
echo ""
echo "2. 选择应用: my-qbusiness-app"
echo ""
echo "3. Access management → Add users and groups"
echo ""
echo "4. 选择用户: qbusiness-admin"
echo ""
echo "5. 选择订阅层级并分配"
echo ""
echo "详细步骤请查看: GRANT_ACCESS_STEPS.md"
echo ""
echo "=========================================="
