#!/bin/bash

set -e

echo "=========================================="
echo "Amazon Q Business Deployment via AWS CLI"
echo "=========================================="
echo ""

# Check prerequisites
echo "Checking prerequisites..."

if ! command -v aws &> /dev/null; then
    echo "Error: AWS CLI is not installed"
    exit 1
fi

if ! command -v jq &> /dev/null; then
    echo "Error: jq is not installed"
    exit 1
fi

# Configuration
APP_NAME="${1:-my-qbusiness-app}"
REGION="${2:-eu-central-1}"
DESCRIPTION="Amazon Q Business Application for enterprise search and chat"

echo "Configuration:"
echo "  Application Name: $APP_NAME"
echo "  Region: $REGION"
echo ""

# Check for IAM Identity Center
echo "Checking IAM Identity Center..."
# Try common regions where Identity Center might be configured
for CHECK_REGION in $REGION us-east-1 eu-north-1 us-west-2; do
    IDENTITY_CENTER=$(aws sso-admin list-instances --region $CHECK_REGION 2>&1) || true
    if echo "$IDENTITY_CENTER" | grep -q "InstanceArn"; then
        INSTANCE_ARN=$(echo "$IDENTITY_CENTER" | jq -r '.Instances[0].InstanceArn')
        echo "  Found Identity Center: $INSTANCE_ARN (checked in region: $CHECK_REGION)"
        break
    fi
done

if [ -z "$INSTANCE_ARN" ]; then
    echo ""
    echo "ERROR: AWS IAM Identity Center is not enabled in your account."
    echo ""
    echo "Amazon Q Business requires IAM Identity Center. To set it up:"
    echo "  1. Go to https://console.aws.amazon.com/singlesignon"
    echo "  2. Click 'Enable' to activate IAM Identity Center"
    echo "  3. Complete the setup wizard"
    echo "  4. Run this script again"
    echo ""
    exit 1
fi

# Create IAM Role for Q Business
echo ""
echo "Creating IAM role for Amazon Q Business..."

ROLE_NAME="${APP_NAME}-app-role"
TRUST_POLICY='{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "qbusiness.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}'

# Check if role exists
if aws iam get-role --role-name "$ROLE_NAME" --region $REGION 2>/dev/null; then
    echo "  Role $ROLE_NAME already exists, using existing role"
    ROLE_ARN=$(aws iam get-role --role-name "$ROLE_NAME" --region $REGION | jq -r '.Role.Arn')
else
    ROLE_ARN=$(aws iam create-role \
        --role-name "$ROLE_NAME" \
        --assume-role-policy-document "$TRUST_POLICY" \
        --region $REGION | jq -r '.Role.Arn')
    echo "  Created role: $ROLE_ARN"

    # Attach basic policy
    aws iam put-role-policy \
        --role-name "$ROLE_NAME" \
        --policy-name "${APP_NAME}-policy" \
        --policy-document '{
          "Version": "2012-10-17",
          "Statement": [
            {
              "Effect": "Allow",
              "Action": [
                "qbusiness:*",
                "s3:GetObject",
                "s3:ListBucket"
              ],
              "Resource": "*"
            }
          ]
        }' --region $REGION

    echo "  Waiting for role to propagate..."
    sleep 10
fi

# Create Q Business Application
echo ""
echo "Creating Amazon Q Business application..."

APP_RESULT=$(aws qbusiness create-application \
    --display-name "$APP_NAME" \
    --description "$DESCRIPTION" \
    --role-arn "$ROLE_ARN" \
    --identity-center-instance-arn "$INSTANCE_ARN" \
    --region $REGION 2>&1) || true

if echo "$APP_RESULT" | grep -q "applicationId"; then
    APP_ID=$(echo "$APP_RESULT" | jq -r '.applicationId')
    APP_ARN=$(echo "$APP_RESULT" | jq -r '.applicationArn')
    echo "  Created application: $APP_ID"
    echo "$APP_RESULT" > /tmp/qbusiness-app.json
else
    echo "Error creating application:"
    echo "$APP_RESULT"
    exit 1
fi

# Create Index
echo ""
echo "Creating Amazon Q Business index..."

INDEX_RESULT=$(aws qbusiness create-index \
    --application-id "$APP_ID" \
    --display-name "${APP_NAME}-index" \
    --type ENTERPRISE \
    --capacity-configuration units=1 \
    --region $REGION 2>&1) || true

if echo "$INDEX_RESULT" | grep -q "indexId"; then
    INDEX_ID=$(echo "$INDEX_RESULT" | jq -r '.indexId')
    echo "  Created index: $INDEX_ID"
    echo "$INDEX_RESULT" > /tmp/qbusiness-index.json
else
    echo "Error creating index:"
    echo "$INDEX_RESULT"
fi

# Create Web Experience
echo ""
echo "Creating web experience..."

WEB_RESULT=$(aws qbusiness create-web-experience \
    --application-id "$APP_ID" \
    --title "$APP_NAME" \
    --subtitle "Your AI-powered business assistant" \
    --region $REGION 2>&1) || true

if echo "$WEB_RESULT" | grep -q "webExperienceId"; then
    WEB_EXP_ID=$(echo "$WEB_RESULT" | jq -r '.webExperienceId')
    WEB_URL=$(echo "$WEB_RESULT" | jq -r '.webExperienceArn')
    echo "  Created web experience: $WEB_EXP_ID"
    echo "$WEB_RESULT" > /tmp/qbusiness-web-experience.json
else
    echo "Error creating web experience:"
    echo "$WEB_RESULT"
fi

echo ""
echo "=========================================="
echo "Deployment Complete!"
echo "=========================================="
echo ""
echo "Application ID: $APP_ID"
echo "Index ID: $INDEX_ID"
echo "Web Experience ID: $WEB_EXP_ID"
echo ""
echo "Configuration files saved to:"
echo "  - /tmp/qbusiness-app.json"
echo "  - /tmp/qbusiness-index.json"
echo "  - /tmp/qbusiness-web-experience.json"
echo ""
echo "Next steps:"
echo "  1. Configure user access via IAM Identity Center"
echo "  2. Set up data sources (S3, SharePoint, etc.)"
echo "  3. Configure retrieval settings"
echo "  4. Test the web experience"
echo ""
echo "To get the web experience URL:"
echo "  jq -r '.defaultEndpoint' /tmp/qbusiness-web-experience.json"
echo ""
