# ============================================
# Amazon Q Business Chat Session via Terraform
# 使用 Terraform 启动和管理聊天会话
# ============================================

# 变量定义 - 聊天会话相关
variable "enable_chat_session" {
  description = "Enable chat session creation"
  type        = bool
  default     = false
}

variable "chat_user_message" {
  description = "User message to send to Q Business"
  type        = string
  default     = "What is our remote work policy?"
}

variable "chat_user_id" {
  description = "IAM Identity Center user ID for chat"
  type        = string
  default     = "108cc92c-b0a1-7073-1ac0-b153f32f9e6c"  # qbusiness-admin user ID
}

# ============================================
# 启动聊天会话
# ============================================
resource "null_resource" "qbusiness_chat_session" {
  count = var.enable_chat_session ? 1 : 0

  triggers = {
    user_message = var.chat_user_message
    timestamp    = timestamp()
  }

  provisioner "local-exec" {
    command = <<-EOT
      # 获取应用 ID
      APP_ID="8ea6e006-85cc-47bd-a587-b14bff19884d"
      USER_ID="${var.chat_user_id}"
      REGION="us-west-2"

      # 发送消息并保存响应
      aws qbusiness chat-sync \
        --application-id $APP_ID \
        --user-id $USER_ID \
        --user-message "${var.chat_user_message}" \
        --region $REGION \
        > /tmp/qbusiness-chat-response.json

      # 显示响应
      echo "========================================"
      echo "Q Business Response:"
      echo "========================================"
      cat /tmp/qbusiness-chat-response.json | jq -r '.systemMessage'
      echo ""
      echo "========================================"

      # 保存会话 ID 以便后续使用
      cat /tmp/qbusiness-chat-response.json | jq -r '.conversationId' > /tmp/qbusiness-conversation-id.txt
    EOT
  }
}

# ============================================
# 持续会话（多轮对话）
# ============================================
resource "null_resource" "qbusiness_followup_message" {
  count = var.enable_chat_session ? 1 : 0

  triggers = {
    depends_on_chat = null_resource.qbusiness_chat_session[0].id
  }

  provisioner "local-exec" {
    command = <<-EOT
      APP_ID="8ea6e006-85cc-47bd-a587-b14bff19884d"
      USER_ID="${var.chat_user_id}"
      REGION="us-west-2"

      # 读取之前的会话 ID
      CONVERSATION_ID=$(cat /tmp/qbusiness-conversation-id.txt 2>/dev/null || echo "")

      if [ -n "$CONVERSATION_ID" ]; then
        echo "继续之前的会话: $CONVERSATION_ID"

        # 发送后续消息
        aws qbusiness chat-sync \
          --application-id $APP_ID \
          --user-id $USER_ID \
          --conversation-id $CONVERSATION_ID \
          --user-message "Tell me more details" \
          --region $REGION \
          > /tmp/qbusiness-followup-response.json

        echo "========================================"
        echo "Follow-up Response:"
        echo "========================================"
        cat /tmp/qbusiness-followup-response.json | jq -r '.systemMessage'
      else
        echo "No previous conversation found"
      fi
    EOT
  }

  depends_on = [null_resource.qbusiness_chat_session]
}

# ============================================
# 输出聊天会话信息
# ============================================
output "chat_session_info" {
  description = "Chat session information"
  value = var.enable_chat_session ? {
    user_message     = var.chat_user_message
    response_file    = "/tmp/qbusiness-chat-response.json"
    conversation_id  = "Check /tmp/qbusiness-conversation-id.txt"
    user_id          = var.chat_user_id
  } : null
}

# ============================================
# 数据源：读取聊天响应
# ============================================
data "external" "chat_response" {
  count = var.enable_chat_session ? 1 : 0

  program = ["bash", "-c", <<-EOT
    if [ -f /tmp/qbusiness-chat-response.json ]; then
      jq -n --arg msg "$(cat /tmp/qbusiness-chat-response.json | jq -r '.systemMessage // "No response"')" \
            --arg conv "$(cat /tmp/qbusiness-conversation-id.txt 2>/dev/null || echo 'none')" \
            '{message: $msg, conversation_id: $conv}'
    else
      echo '{"message": "No response yet", "conversation_id": "none"}'
    fi
  EOT
  ]

  depends_on = [null_resource.qbusiness_chat_session]
}

output "chat_response" {
  description = "Latest chat response from Q Business"
  value       = var.enable_chat_session ? data.external.chat_response[0].result : null
}
