#!/bin/bash

# Amazon Q Business Interactive Chat Session
# 交互式聊天会话脚本

APP_ID="8ea6e006-85cc-47bd-a587-b14bff19884d"
USER_ID="108cc92c-b0a1-7073-1ac0-b153f32f9e6c"
REGION="us-west-2"
CONVERSATION_ID=""
PARENT_MESSAGE_ID=""

# 颜色定义
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${GREEN}======================================${NC}"
echo -e "${GREEN}Amazon Q Business 交互式会话${NC}"
echo -e "${GREEN}======================================${NC}"
echo -e "${YELLOW}命令:${NC}"
echo -e "  输入问题进行对话"
echo -e "  'exit' 或 'quit' - 退出"
echo -e "  'clear' - 清空会话历史"
echo -e "  'help' - 显示帮助"
echo -e "${GREEN}======================================${NC}"
echo ""

# 发送消息函数
send_message() {
    local message="$1"
    local params="--application-id $APP_ID --user-id $USER_ID --region $REGION"

    # 如果有会话 ID，添加到参数中
    if [ -n "$CONVERSATION_ID" ]; then
        params="$params --conversation-id $CONVERSATION_ID"
    fi

    if [ -n "$PARENT_MESSAGE_ID" ]; then
        params="$params --parent-message-id $PARENT_MESSAGE_ID"
    fi

    # 发送消息并解析响应
    local response=$(aws qbusiness chat-sync \
        $params \
        --user-message "$message" \
        2>&1)

    if [ $? -eq 0 ]; then
        # 保存会话 ID 以保持上下文
        CONVERSATION_ID=$(echo "$response" | jq -r '.conversationId // empty')
        PARENT_MESSAGE_ID=$(echo "$response" | jq -r '.systemMessageId // empty')

        # 显示回复
        echo -e "${BLUE}Q Business:${NC}"
        echo "$response" | jq -r '.systemMessage // .failedAttachments // "No response"'

        # 显示引用的文档（如果有）
        local sources=$(echo "$response" | jq -r '.sourceAttributions[]?.title // empty' 2>/dev/null)
        if [ -n "$sources" ]; then
            echo -e "\n${YELLOW}参考文档:${NC}"
            echo "$response" | jq -r '.sourceAttributions[] | "  - \(.title) [\(.snippet // "")]"'
        fi
    else
        echo -e "${YELLOW}错误: 无法发送消息${NC}"
        echo "$response"
    fi

    echo ""
}

# 显示帮助
show_help() {
    echo -e "${YELLOW}可用命令:${NC}"
    echo "  exit/quit  - 退出聊天"
    echo "  clear      - 清空会话历史（开始新对话）"
    echo "  help       - 显示此帮助信息"
    echo "  status     - 显示当前会话状态"
    echo ""
}

# 显示状态
show_status() {
    echo -e "${YELLOW}当前会话状态:${NC}"
    echo "  Application ID: $APP_ID"
    echo "  User ID: $USER_ID"
    echo "  Region: $REGION"
    echo "  Conversation ID: ${CONVERSATION_ID:-'(new session)'}"
    echo ""
}

# 主循环
while true; do
    echo -n -e "${GREEN}You:${NC} "
    read -r input

    # 处理空输入
    if [ -z "$input" ]; then
        continue
    fi

    # 处理命令
    case "$input" in
        exit|quit)
            echo -e "${GREEN}再见! 会话已结束。${NC}"
            break
            ;;
        clear)
            CONVERSATION_ID=""
            PARENT_MESSAGE_ID=""
            echo -e "${YELLOW}会话历史已清空，开始新对话${NC}"
            echo ""
            ;;
        help)
            show_help
            ;;
        status)
            show_status
            ;;
        *)
            send_message "$input"
            ;;
    esac
done
