# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

This repository contains Terraform configurations for deploying Amazon Q Business, an AI-powered enterprise search and chat application. The deployment uses a **hybrid approach**: Terraform manages IAM resources while AWS CLI commands (via `null_resource` provisioners) handle Amazon Q Business resources, since native Terraform provider support for Q Business is not yet available.

## Key Architecture Patterns

### Hybrid Terraform + AWS CLI Approach

The deployment combines:
- **Terraform resources**: IAM roles, policies, and orchestration
- **null_resource provisioners**: Execute AWS CLI commands for Q Business resources
- **data.external**: Extract resource IDs from JSON files written to `/tmp/`
- **Local state files**: `/tmp/qbusiness-*.json` files persist resource metadata between runs

Key pattern used throughout:
```hcl
resource "null_resource" "qbusiness_application" {
  provisioner "local-exec" {
    command = <<-EOT
      # Check if resource exists
      EXISTING=$(aws qbusiness list-applications ...)
      if [ -z "$EXISTING" ]; then
        # Create new
        aws qbusiness create-application ... > /tmp/qbusiness-app.json
      else
        # Get existing
        aws qbusiness get-application ... > /tmp/qbusiness-app.json
      fi
    EOT
  }
}
```

### Resource Dependencies

1. **IAM Role** (`aws_iam_role.qbusiness_app_role`) → Created first
2. **Q Business Application** (`null_resource.qbusiness_application`) → Uses IAM role
3. **Index** (`null_resource.qbusiness_index`) → Requires application ID
4. **Web Experience** (`null_resource.qbusiness_web_experience`) → Requires index

All Q Business resources extract IDs from `/tmp/qbusiness-*.json` files using `jq`.

## Essential Commands

### Deployment
```bash
# Standard Terraform workflow
terraform init
terraform plan
terraform apply

# Quick deployment via script
./deploy.sh [app-name] [region]

# Example
./deploy.sh my-app us-west-2
```

### Chat Sessions

Three ways to interact with deployed Q Business:

1. **Simple one-off questions**:
   ```bash
   ./chat-simple.sh 'What is our remote work policy?'
   ```

2. **Interactive multi-turn conversation**:
   ```bash
   ./qbusiness-interactive.sh
   ```

3. **Terraform-managed chat session**:
   ```bash
   terraform apply -var="enable_chat_session=true" -var='chat_user_message=Your question'
   ```

### Inspection Commands
```bash
# Get application details
APP_ID=$(jq -r '.applicationId' /tmp/qbusiness-app.json)
aws qbusiness get-application --application-id $APP_ID --region us-west-2

# Check index status and statistics
INDEX_ID=$(jq -r '.indexId' /tmp/qbusiness-index.json)
aws qbusiness get-index --application-id $APP_ID --index-id $INDEX_ID --region us-west-2

# List data sources
aws qbusiness list-data-sources --application-id $APP_ID --index-id $INDEX_ID --region us-west-2

# Check data source sync status
aws qbusiness list-data-source-sync-jobs --application-id $APP_ID --index-id $INDEX_ID --data-source-id <DATA_SOURCE_ID> --region us-west-2

# Get web experience URL
jq -r '.defaultEndpoint' /tmp/qbusiness-web-experience.json
```

### User Management

**Current Identity Center Configuration**:
- Instance Name: `iam_claude`
- Identity Store ID: `d-c367615483`
- Region: `eu-north-1` (Stockholm)
- Existing User: `qbusiness-admin` (ID: `108cc92c-b0a1-7073-1ac0-b153f32f9e6c`)

```bash
# List Identity Center users (note: Identity Center is in eu-north-1)
IDENTITY_STORE_ID="d-c367615483"
aws identitystore list-users --identity-store-id $IDENTITY_STORE_ID --region eu-north-1

# Get specific user
aws identitystore get-user \
  --identity-store-id $IDENTITY_STORE_ID \
  --user-id 108cc92c-b0a1-7073-1ac0-b153f32f9e6c \
  --region eu-north-1

# Create new user
aws identitystore create-user \
  --identity-store-id $IDENTITY_STORE_ID \
  --user-name "username" \
  --display-name "Display Name" \
  --name Formatted="Name" FamilyName="Last" GivenName="First" \
  --emails Value="user@example.com",Primary=true \
  --region eu-north-1

# Update user
aws identitystore update-user \
  --identity-store-id $IDENTITY_STORE_ID \
  --user-id <USER_ID> \
  --operations Op=replace,Path=/displayName,Value="New Name" \
  --region eu-north-1
```

**Important**: After creating users in Identity Center, you MUST assign them to the Q Business application through the AWS Console:
1. Go to Amazon Q Business console (us-west-2 region)
2. Select your application
3. Access management → Add users and groups
4. Select the user and choose a subscription tier (Lite or Pro)
5. Click Assign

### Cleanup
```bash
terraform destroy
```

## Critical Prerequisites

1. **AWS IAM Identity Center MUST be enabled** - Q Business cannot function without it
   - Check: `./get-identity-center-arn.sh`
   - Enable at: https://console.aws.amazon.com/singlesignon

2. **Required tools**:
   - Terraform >= 1.0
   - AWS CLI v2
   - jq (JSON processor)

3. **Region availability**: Q Business is only available in limited regions (us-east-1, us-west-2, etc.)

## Important File Locations

- **Terraform state**: `terraform.tfstate` (local)
- **Resource metadata**: `/tmp/qbusiness-*.json` (referenced by data sources)
- **Conversation IDs**: `/tmp/qbusiness-conversation-id.txt` (for multi-turn chat)
- **Configuration**: `terraform.tfvars` (deployment settings)
- **Chat variables**: `terraform-chat.tfvars` (chat session settings)

## Configuration Variables

Key variables in `variables.tf`:

- `identity_center_instance_arn` - **REQUIRED** - ARN from Identity Center
- `application_name` - Base name for all resources
- `aws_region` - Deployment region (must support Q Business)
- `index_capacity_units` - Index capacity (1 for dev/testing, higher for prod)
- `create_s3_datasource` - Whether to create S3 data source
- `data_source_bucket` - S3 bucket name for data source
- `enable_chat_session` - Enable Terraform chat session (in `qbusiness-chat-session.tf`)
- `chat_user_id` - IAM Identity Center user ID for chat

## Modular File Structure

- **main.tf**: Core resources (IAM roles, Q Business app, index, web experience)
- **qbusiness-chat-session.tf**: Chat session management via Terraform
- **archive.tf**: Project backup and distribution archives
- **variables.tf**: Input variable definitions
- **outputs.tf**: Output values
- **archive-examples.tf**: Example archive configurations (commented out)

## Common Development Patterns

### Testing Changes
Since state is stored in `/tmp/` JSON files, you can:
1. Manually inspect: `cat /tmp/qbusiness-app.json | jq`
2. Force recreation: Delete `/tmp/qbusiness-*.json` and run `terraform apply`
3. Import existing resources: Update JSON files with existing resource IDs

### Adding New Data Sources
Extend the pattern in `main.tf` by adding a new `null_resource` that:
1. Reads `APP_ID` and `INDEX_ID` from `/tmp/qbusiness-*.json`
2. Calls `aws qbusiness create-data-source`
3. Saves result to `/tmp/qbusiness-datasource-*.json`

### Handling Region Changes
When changing regions, remember to:
1. Update `terraform.tfvars`
2. Verify Q Business availability in target region
3. Delete `/tmp/qbusiness-*.json` files (they reference old region)
4. Run `terraform apply` to recreate in new region

## Troubleshooting Tips

1. **"Resource not found" errors**: Check if `/tmp/qbusiness-*.json` files exist and contain valid IDs
2. **"Identity Center not enabled"**: Run `./get-identity-center-arn.sh` to verify setup
3. **Chat authentication errors**: Verify user ID exists in Identity Center with `aws identitystore get-user`
4. **Empty responses**: Check index statistics and ensure data sources have synced
5. **Terraform state drift**: The hybrid approach means Terraform can't detect manual AWS CLI changes
