# 使用 Terraform 启动 Amazon Q Business 会话

## 快速开始

### 方法 1: 使用配置文件启动会话

1. **编辑聊天配置**

编辑 `terraform-chat.tfvars` 文件：

```hcl
# 启用聊天会话
enable_chat_session = true

# 你想问的问题
chat_user_message = "What is our remote work policy?"

# 用户 ID
chat_user_id = "108cc92c-b0a1-7073-1ac0-b153f32f9e6c"
```

2. **运行 Terraform**

```bash
# 初始化（如果还没有）
terraform init

# 预览变更
terraform plan -var-file="terraform-chat.tfvars"

# 应用配置（启动会话）
terraform apply -var-file="terraform-chat.tfvars" -auto-approve
```

3. **查看响应**

Terraform 会自动显示 Q Business 的回复，也可以手动查看：

```bash
# 查看完整响应
cat /tmp/qbusiness-chat-response.json | jq

# 仅查看回复消息
cat /tmp/qbusiness-chat-response.json | jq -r '.systemMessage'

# 查看会话 ID
cat /tmp/qbusiness-conversation-id.txt
```

---

## 方法 2: 命令行直接指定问题

不需要编辑配置文件，直接在命令行中指定：

```bash
terraform apply \
  -var="enable_chat_session=true" \
  -var='chat_user_message=How many vacation days do I get?' \
  -auto-approve
```

---

## 方法 3: 多轮对话

### 第一轮对话

```bash
terraform apply \
  -var="enable_chat_session=true" \
  -var='chat_user_message=What is CloudSync Pro?' \
  -auto-approve
```

### 查看会话 ID 并继续对话

```bash
# 获取会话 ID
CONVERSATION_ID=$(cat /tmp/qbusiness-conversation-id.txt)
echo "Conversation ID: $CONVERSATION_ID"

# 继续对话（使用相同的会话 ID）
aws qbusiness chat-sync \
  --application-id 8ea6e006-85cc-47bd-a587-b14bff19884d \
  --user-id 108cc92c-b0a1-7073-1ac0-b153f32f9e6c \
  --conversation-id $CONVERSATION_ID \
  --user-message "What are the pricing tiers?" \
  --region us-west-2 | jq -r '.systemMessage'
```

---

## 实用示例

### 示例 1: 询问公司政策

```bash
terraform apply \
  -var="enable_chat_session=true" \
  -var='chat_user_message=What is our remote work policy?' \
  -auto-approve
```

### 示例 2: 询问员工福利

```bash
terraform apply \
  -var="enable_chat_session=true" \
  -var='chat_user_message=Tell me about our 401k benefits' \
  -auto-approve
```

### 示例 3: 产品相关问题

```bash
terraform apply \
  -var="enable_chat_session=true" \
  -var='chat_user_message=What integrations does CloudSync Pro support?' \
  -auto-approve
```

---

## 高级用法：创建交互式会话脚本

创建一个脚本来简化 Terraform 聊天：

```bash
#!/bin/bash
# 保存为 terraform-chat.sh

echo "Amazon Q Business - Terraform Chat Session"
echo "=========================================="
echo ""

while true; do
    echo -n "Your question (or 'exit' to quit): "
    read question

    if [ "$question" = "exit" ]; then
        echo "Goodbye!"
        break
    fi

    echo ""
    echo "Asking Q Business..."

    terraform apply \
      -var="enable_chat_session=true" \
      -var="chat_user_message=$question" \
      -auto-approve > /dev/null 2>&1

    echo ""
    echo "Q Business Response:"
    echo "----------------------------------------"
    cat /tmp/qbusiness-chat-response.json | jq -r '.systemMessage'
    echo "----------------------------------------"
    echo ""
done
```

使用方法：

```bash
chmod +x terraform-chat.sh
./terraform-chat.sh
```

---

## 配置选项说明

### `qbusiness-chat-session.tf` 变量

| 变量 | 说明 | 默认值 |
|------|------|--------|
| `enable_chat_session` | 是否启用聊天会话 | `false` |
| `chat_user_message` | 发送给 Q Business 的消息 | `"What is our remote work policy?"` |
| `chat_user_id` | IAM Identity Center 用户 ID | `"108cc92c-b0a1-7073-1ac0-b153f32f9e6c"` |

### 输出

Terraform 会输出：

```hcl
chat_session_info = {
  conversation_id = "Check /tmp/qbusiness-conversation-id.txt"
  response_file = "/tmp/qbusiness-chat-response.json"
  user_id = "108cc92c-b0a1-7073-1ac0-b153f32f9e6c"
  user_message = "What is our remote work policy?"
}

chat_response = {
  conversation_id = "xxx-xxx-xxx"
  message = "Based on our company policy..."
}
```

---

## 查看会话历史

```bash
# 查看最近的响应
cat /tmp/qbusiness-chat-response.json | jq

# 查看后续消息响应
cat /tmp/qbusiness-followup-response.json | jq

# 提取关键信息
jq -r '.systemMessage' /tmp/qbusiness-chat-response.json
jq -r '.conversationId' /tmp/qbusiness-chat-response.json
jq -r '.sourceAttributions[]?.title' /tmp/qbusiness-chat-response.json
```

---

## 禁用聊天会话

如果不想每次 `terraform apply` 都触发聊天：

```bash
# 方法 1: 在 tfvars 中设置
enable_chat_session = false

# 方法 2: 命令行覆盖
terraform apply -var="enable_chat_session=false"
```

---

## 集成到现有 Terraform 工作流

### 在 `terraform.tfvars` 中添加

```hcl
# 保持聊天默认禁用
enable_chat_session = false
```

### 仅在需要时启用

```bash
# 正常部署（不触发聊天）
terraform apply

# 需要聊天时
terraform apply -var="enable_chat_session=true" -var='chat_user_message=Your question here'
```

---

## 故障排查

### 问题 1: 没有响应

检查用户权限：

```bash
aws qbusiness get-application \
  --application-id 8ea6e006-85cc-47bd-a587-b14bff19884d \
  --region us-west-2
```

### 问题 2: 认证错误

确保用户 ID 正确：

```bash
# 列出所有用户
aws identitystore list-users \
  --identity-store-id d-9067e46e40 \
  --region us-west-2
```

### 问题 3: 文件不存在

确保创建了必要的临时文件：

```bash
mkdir -p /tmp
touch /tmp/qbusiness-chat-response.json
```

---

## 示例输出

成功执行后的输出：

```
null_resource.qbusiness_chat_session[0]: Creating...
null_resource.qbusiness_chat_session[0]: Provisioning with 'local-exec'...
null_resource.qbusiness_chat_session[0] (local-exec): ========================================
null_resource.qbusiness_chat_session[0] (local-exec): Q Business Response:
null_resource.qbusiness_chat_session[0] (local-exec): ========================================
null_resource.qbusiness_chat_session[0] (local-exec): Based on our company remote work policy,
null_resource.qbusiness_chat_session[0] (local-exec): employees are allowed to work remotely...
null_resource.qbusiness_chat_session[0] (local-exec): ========================================
null_resource.qbusiness_chat_session[0]: Creation complete after 2s

Outputs:

chat_response = {
  "conversation_id" = "abc123..."
  "message" = "Based on our company remote work policy..."
}
```

---

## 最佳实践

1. **默认禁用聊天会话** - 避免每次 `terraform apply` 都触发
2. **使用有意义的变量名** - 便于理解和维护
3. **保存会话 ID** - 用于多轮对话
4. **查看源引用** - 了解答案来自哪些文档
5. **定期清理临时文件** - 避免过期数据干扰

---

## 资源

- **Terraform 文件**: `qbusiness-chat-session.tf`
- **配置文件**: `terraform-chat.tfvars`
- **响应存储**: `/tmp/qbusiness-chat-response.json`
- **会话 ID**: `/tmp/qbusiness-conversation-id.txt`

---

**提示**: 对于频繁交互，建议使用之前创建的 `qbusiness-interactive.sh` 脚本，它提供更好的交互体验。Terraform 方法更适合自动化和集成到 CI/CD 流程中。
