# ============================================
# Amazon Q Business Project Archive
# 用于备份和分发 Terraform 配置
# ============================================

# Note: archive provider is declared in main.tf

# 创建 archives 目录（如果不存在）
resource "null_resource" "create_archives_dir" {
  provisioner "local-exec" {
    command = "mkdir -p ${path.module}/archives"
  }

  triggers = {
    always_run = timestamp()
  }
}

# ============================================
# 完整项目归档
# ============================================
data "archive_file" "project_backup" {
  type        = "zip"
  source_dir  = path.module
  output_path = "${path.module}/archives/amazon-q-terraform-backup-${formatdate("YYYY-MM-DD", timestamp())}.zip"

  excludes = [
    ".terraform/**",
    ".terraform.lock.hcl",
    "*.tfstate",
    "*.tfstate.*",
    ".git/**",
    "archives/**",
    ".gitignore"
  ]

  depends_on = [null_resource.create_archives_dir]
}

# ============================================
# 仅 Terraform 配置文件
# ============================================
data "archive_file" "terraform_only" {
  type        = "zip"
  output_path = "${path.module}/archives/terraform-configs.zip"

  source {
    content  = file("${path.module}/main.tf")
    filename = "main.tf"
  }

  source {
    content  = file("${path.module}/variables.tf")
    filename = "variables.tf"
  }

  source {
    content  = file("${path.module}/outputs.tf")
    filename = "outputs.tf"
  }

  source {
    content  = file("${path.module}/terraform.tfvars")
    filename = "terraform.tfvars"
  }

  depends_on = [null_resource.create_archives_dir]
}

# ============================================
# 文档归档
# ============================================
data "archive_file" "documentation_bundle" {
  type        = "zip"
  output_path = "${path.module}/archives/documentation.zip"

  source {
    content  = fileexists("${path.module}/README.md") ? file("${path.module}/README.md") : "No README"
    filename = "README.md"
  }

  source {
    content  = fileexists("${path.module}/DEPLOYMENT_GUIDE.md") ? file("${path.module}/DEPLOYMENT_GUIDE.md") : "No guide"
    filename = "DEPLOYMENT_GUIDE.md"
  }

  source {
    content  = fileexists("${path.module}/SETUP_COMPLETE.md") ? file("${path.module}/SETUP_COMPLETE.md") : "No setup"
    filename = "SETUP_COMPLETE.md"
  }

  source {
    content  = fileexists("${path.module}/DEPLOYMENT_SUMMARY.md") ? file("${path.module}/DEPLOYMENT_SUMMARY.md") : "No summary"
    filename = "DEPLOYMENT_SUMMARY.md"
  }

  depends_on = [null_resource.create_archives_dir]
}

# ============================================
# 脚本归档
# ============================================
data "archive_file" "scripts_bundle" {
  type        = "zip"
  output_path = "${path.module}/archives/scripts.zip"

  source {
    content  = fileexists("${path.module}/deploy.sh") ? file("${path.module}/deploy.sh") : "#!/bin/bash\necho 'No deploy script'"
    filename = "deploy.sh"
  }

  source {
    content  = fileexists("${path.module}/qbusiness-interactive.sh") ? file("${path.module}/qbusiness-interactive.sh") : "#!/bin/bash\necho 'No interactive script'"
    filename = "qbusiness-interactive.sh"
  }

  source {
    content  = fileexists("${path.module}/get-identity-center-arn.sh") ? file("${path.module}/get-identity-center-arn.sh") : "#!/bin/bash\necho 'No script'"
    filename = "get-identity-center-arn.sh"
  }

  depends_on = [null_resource.create_archives_dir]
}

# ============================================
# Outputs
# ============================================
output "archive_info" {
  description = "Information about created archives"
  value = {
    project_backup = {
      path = data.archive_file.project_backup.output_path
      size = "${formatdate("YYYY-MM-DD", timestamp())} - ${data.archive_file.project_backup.output_size} bytes"
      md5  = data.archive_file.project_backup.output_md5
    }
    terraform_configs = {
      path = data.archive_file.terraform_only.output_path
      size = "${data.archive_file.terraform_only.output_size} bytes"
      md5  = data.archive_file.terraform_only.output_md5
    }
    documentation = {
      path = data.archive_file.documentation_bundle.output_path
      size = "${data.archive_file.documentation_bundle.output_size} bytes"
      md5  = data.archive_file.documentation_bundle.output_md5
    }
    scripts = {
      path = data.archive_file.scripts_bundle.output_path
      size = "${data.archive_file.scripts_bundle.output_size} bytes"
      md5  = data.archive_file.scripts_bundle.output_md5
    }
  }
}
