terraform {
  required_version = ">= 1.0"

  required_providers {
    aws = {
      source  = "hashicorp/aws"
      version = "~> 5.0"
    }
    null = {
      source  = "hashicorp/null"
      version = "~> 3.0"
    }
    archive = {
      source  = "hashicorp/archive"
      version = "~> 2.4"
    }
  }
}

provider "aws" {
  region = var.aws_region
}

# IAM Role for Amazon Q Business Application
resource "aws_iam_role" "qbusiness_app_role" {
  name = "${var.application_name}-app-role"

  assume_role_policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Action = "sts:AssumeRole"
        Effect = "Allow"
        Principal = {
          Service = "qbusiness.amazonaws.com"
        }
      }
    ]
  })

  tags = var.tags
}

# IAM Policy for Amazon Q Business Application
resource "aws_iam_role_policy" "qbusiness_app_policy" {
  name = "${var.application_name}-app-policy"
  role = aws_iam_role.qbusiness_app_role.id

  policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Effect = "Allow"
        Action = [
          "qbusiness:*",
          "s3:GetObject",
          "s3:ListBucket"
        ]
        Resource = "*"
      }
    ]
  })
}

# Amazon Q Business Application (using AWS CLI via null_resource)
resource "null_resource" "qbusiness_application" {
  triggers = {
    application_name = var.application_name
    role_arn         = aws_iam_role.qbusiness_app_role.arn
    description      = var.application_description
  }

  provisioner "local-exec" {
    command = <<-EOT
      # Check if application already exists
      EXISTING_APP=$(aws qbusiness list-applications --region ${var.aws_region} | \
        jq -r '.applications[] | select(.displayName == "${var.application_name}") | .applicationId' | head -1)

      if [ -z "$EXISTING_APP" ]; then
        # Create new application
        aws qbusiness create-application \
          --display-name "${var.application_name}" \
          --description "${var.application_description}" \
          --region ${var.aws_region} \
          --role-arn ${aws_iam_role.qbusiness_app_role.arn} \
          --identity-center-instance-arn ${var.identity_center_instance_arn} > /tmp/qbusiness-app.json
      else
        # Get existing application details
        aws qbusiness get-application \
          --application-id $EXISTING_APP \
          --region ${var.aws_region} > /tmp/qbusiness-app.json
      fi
    EOT
  }

  provisioner "local-exec" {
    when    = destroy
    command = <<-EOT
      APP_ID=$(cat /tmp/qbusiness-app.json 2>/dev/null | jq -r '.applicationId // empty')
      if [ ! -z "$APP_ID" ]; then
        aws qbusiness delete-application --application-id $APP_ID || true
      fi
    EOT
  }

  depends_on = [aws_iam_role.qbusiness_app_role]
}

# Extract application ID from the created application
data "external" "application_id" {
  program = ["bash", "-c", "jq -n --arg id \"$(cat /tmp/qbusiness-app.json 2>/dev/null | jq -r '.applicationId // \"not-created\"')\" '{\"application_id\": $id}'"]

  depends_on = [null_resource.qbusiness_application]
}

# Amazon Q Business Index
resource "null_resource" "qbusiness_index" {
  triggers = {
    application_id = data.external.application_id.result.application_id
    index_name     = "${var.application_name}-index"
  }

  provisioner "local-exec" {
    command = <<-EOT
      APP_ID=$(cat /tmp/qbusiness-app.json | jq -r '.applicationId')

      # Check if index already exists
      EXISTING_INDEX=$(aws qbusiness list-indices \
        --application-id $APP_ID \
        --region ${var.aws_region} | jq -r '.indices[] | select(.displayName == "${var.application_name}-index") | .indexId' | head -1)

      if [ -z "$EXISTING_INDEX" ]; then
        # Create new index
        aws qbusiness create-index \
          --application-id $APP_ID \
          --display-name "${var.application_name}-index" \
          --type ENTERPRISE \
          --capacity-configuration units=${var.index_capacity_units} \
          --region ${var.aws_region} > /tmp/qbusiness-index.json
      else
        # Get existing index details
        aws qbusiness get-index \
          --application-id $APP_ID \
          --index-id $EXISTING_INDEX \
          --region ${var.aws_region} > /tmp/qbusiness-index.json
      fi
    EOT
  }

  provisioner "local-exec" {
    when    = destroy
    command = <<-EOT
      APP_ID=$(cat /tmp/qbusiness-app.json 2>/dev/null | jq -r '.applicationId // empty')
      INDEX_ID=$(cat /tmp/qbusiness-index.json 2>/dev/null | jq -r '.indexId // empty')
      if [ ! -z "$APP_ID" ] && [ ! -z "$INDEX_ID" ]; then
        aws qbusiness delete-index --application-id $APP_ID --index-id $INDEX_ID || true
      fi
    EOT
  }

  depends_on = [null_resource.qbusiness_application]
}

# IAM Role for Data Source
resource "aws_iam_role" "qbusiness_datasource_role" {
  name = "${var.application_name}-datasource-role"

  assume_role_policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Action = "sts:AssumeRole"
        Effect = "Allow"
        Principal = {
          Service = "qbusiness.amazonaws.com"
        }
      }
    ]
  })

  tags = var.tags
}

# IAM Policy for Data Source (S3 access)
resource "aws_iam_role_policy" "qbusiness_datasource_policy" {
  name = "${var.application_name}-datasource-policy"
  role = aws_iam_role.qbusiness_datasource_role.id

  policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Effect = "Allow"
        Action = [
          "s3:GetObject",
          "s3:ListBucket",
          "s3:GetBucketLocation"
        ]
        Resource = [
          "arn:aws:s3:::${var.data_source_bucket}",
          "arn:aws:s3:::${var.data_source_bucket}/*"
        ]
      }
    ]
  })
}

# Web Experience (Chat Interface)
resource "null_resource" "qbusiness_web_experience" {
  triggers = {
    application_id = data.external.application_id.result.application_id
    title          = var.application_name
  }

  provisioner "local-exec" {
    command = <<-EOT
      APP_ID=$(cat /tmp/qbusiness-app.json | jq -r '.applicationId')

      # Check if web experience already exists
      EXISTING_WEB_EXP=$(aws qbusiness list-web-experiences \
        --application-id $APP_ID \
        --region ${var.aws_region} | jq -r '.webExperiences[0].webExperienceId // empty')

      if [ -z "$EXISTING_WEB_EXP" ]; then
        # Create new web experience
        aws qbusiness create-web-experience \
          --application-id $APP_ID \
          --title "${var.application_name}" \
          --subtitle "${var.web_experience_subtitle}" \
          --role-arn ${aws_iam_role.qbusiness_app_role.arn} \
          --region ${var.aws_region} > /tmp/qbusiness-web-experience.json
      else
        # Get existing web experience details
        aws qbusiness get-web-experience \
          --application-id $APP_ID \
          --web-experience-id $EXISTING_WEB_EXP \
          --region ${var.aws_region} > /tmp/qbusiness-web-experience.json
      fi
    EOT
  }

  provisioner "local-exec" {
    when    = destroy
    command = <<-EOT
      APP_ID=$(cat /tmp/qbusiness-app.json 2>/dev/null | jq -r '.applicationId // empty')
      WEB_EXP_ID=$(cat /tmp/qbusiness-web-experience.json 2>/dev/null | jq -r '.webExperienceId // empty')
      if [ ! -z "$APP_ID" ] && [ ! -z "$WEB_EXP_ID" ]; then
        aws qbusiness delete-web-experience --application-id $APP_ID --web-experience-id $WEB_EXP_ID || true
      fi
    EOT
  }

  depends_on = [null_resource.qbusiness_index]
}
